import Device from '#models/device';
import { normalizeMessageContent, jidNormalizedUser, isJidGroup, getContentType, isLidUser, } from 'baileys';
import CacheService from '#services/cache';
const MEDIA_TYPE = [
    'imageMessage',
    'videoMessage',
    'audioMessage',
    'documentMessage',
    'stickerMessage',
];
export default async function ({ id, sock, WAMessage }) {
    const normalizedMessages = {
        ...WAMessage,
        message: normalizeMessageContent(WAMessage.message),
    };
    const { key, message, broadcast, ...messages } = normalizedMessages;
    const chat = jidNormalizedUser(key.remoteJid || key.remoteJidAlt);
    const botPn = jidNormalizedUser(sock?.user?.id);
    const botLid = jidNormalizedUser(sock?.user?.lid);
    const isGroup = isJidGroup(chat) || false;
    const groupMetadata = isGroup ? await sock.getGroupCache(chat) : {};
    const senderLid = decodeSender(key, 'lid');
    const senderPn = decodeSender(key, 'pn');
    if (!senderPn)
        return;
    const sender = isGroup ? jidNormalizedUser(key.participant) : jidNormalizedUser(key.remoteJid);
    const mtype = getContentType(message);
    const content = getContentMessage({ mtype, message });
    const quoted = getQuoted({ key, mtype, message, sock });
    let cacheDevice = await CacheService.get(id.split('-')[0]);
    if (!cacheDevice) {
        const device = await Device.query()
            .preload('accessTokens')
            .where('id', id.split('-')[0])
            .first();
        await CacheService.set(id.split('-')[0], device?.toJSON());
        cacheDevice = device?.toJSON();
    }
    if (typeof cacheDevice === 'string') {
        cacheDevice = JSON.parse(cacheDevice);
    }
    return {
        key,
        chat,
        fromMe: key.fromMe,
        pushName: messages.pushName,
        senderLid,
        senderPn,
        botLid,
        botPn,
        token: cacheDevice?.accessTokens[0].token || null,
        webhook: cacheDevice?.webhook || null,
        isGroup,
        isSenderBot: senderLid === botLid,
        isSenderSuperAdmin: isGroup
            ? IsSender(sender, groupMetadata.participants, ['superadmin'])
            : false,
        isSenderAdmin: isGroup
            ? IsSender(sender, groupMetadata.participants, ['superadmin', 'admin'])
            : false,
        isBotSuperAdmin: isGroup
            ? IsSender(groupMetadata.addressingMode === 'lid' ? botLid : botPn, groupMetadata.participants, ['superadmin'])
            : false,
        isBotAdmin: isGroup
            ? IsSender(groupMetadata.addressingMode === 'lid' ? botLid : botPn, groupMetadata.participants, ['superadmin', 'admin'])
            : false,
        groupMetadata,
        mtype,
        msg: content,
        quoted,
    };
}
const decodeSender = (key, type) => {
    let sender = null;
    if (isJidGroup(key.remoteJid)) {
        if (type === 'lid') {
            sender = isLidUser(key.participant) ? key.participant : key.participantLid;
        }
        else if (type === 'pn') {
            sender = !isLidUser(key.participant) ? key.participant : key.participantPn;
        }
    }
    else {
        if (type === 'lid') {
            sender = isLidUser(key.remoteJid) ? key.remoteJid : key.senderLid;
        }
        else if (type === 'pn') {
            sender = !isLidUser(key.remoteJid) ? key.remoteJid : key.senderPn;
        }
    }
    return sender ? jidNormalizedUser(sender) : null;
};
const IsSender = (pnOrLid, participants, admin) => {
    return participants.some((participant) => {
        return ((pnOrLid === jidNormalizedUser(participant.id) ||
            pnOrLid === participant.lid ||
            pnOrLid === participant.pn) &&
            admin.includes(participant.admin));
    });
};
const getContentMessage = ({ mtype, message }) => {
    const text = getTextMessage(message);
    return {
        text,
        textWithoutCommand: text.replace(text.split(' ')[0].trim(), '').trim(),
        command: text.split(' ')[0].trim(),
        mentionedJid: message[mtype]?.contextInfo?.mentionedJid || [],
        expiration: message[mtype]?.contextInfo?.expiration || 0,
        isText: mtype === 'conversation' || mtype === 'extendedTextMessage',
        isSticker: mtype === 'stickerMessage',
        isImage: mtype === 'imageMessage',
        isVideo: mtype === 'videoMessage',
        isAudio: mtype === 'audioMessage',
        isDocument: mtype === 'documentMessage',
        isMedia: MEDIA_TYPE.includes(mtype),
    };
};
const getTextMessage = (message) => {
    return (message?.conversation ||
        message?.extendedTextMessage?.text ||
        message?.imageMessage?.caption ||
        message?.videoMessage?.caption ||
        message?.documentMessage?.caption ||
        message?.documentWithCaptionMessage?.message?.documentMessage?.caption ||
        message?.ephemeralMessage?.message?.conversation ||
        message?.ephemeralMessage?.message?.extendedTextMessage?.text ||
        message?.ephemeralMessage?.message?.imageMessage?.caption ||
        message?.ephemeralMessage?.message?.videoMessage?.caption ||
        message?.ephemeralMessage?.message?.documentWithCaptionMessage?.message?.documentMessage
            ?.caption ||
        message?.editedMessage?.message?.protocolMessage?.editedMessage?.conversation ||
        message?.editedMessage?.message?.protocolMessage?.editedMessage?.extendedTextMessage?.text ||
        '');
};
const getQuoted = ({ key, mtype, message, sock, }) => {
    const Quoted = message[mtype]?.contextInfo?.quotedMessage;
    if (!Quoted)
        return undefined;
    const contextInfo = message[mtype]?.contextInfo;
    const quotedMessage = normalizeMessageContent(Quoted);
    const type = getContentType(quotedMessage);
    const content = getContentMessage({
        mtype: type,
        message: quotedMessage,
    });
    const senderLid = isLidUser(contextInfo.participant) ? contextInfo.participant : undefined;
    const senderPn = !isLidUser(contextInfo.participant) ? contextInfo.participant : undefined;
    return {
        key: {
            remoteJid: isJidGroup(key.remoteJid) ? key.remoteJid : contextInfo.participant,
            fromMe: isLidUser(contextInfo.participant)
                ? contextInfo.participant === jidNormalizedUser(sock.user.lid)
                : contextInfo.participant === jidNormalizedUser(sock.user.id),
            id: contextInfo?.stanzaId,
            senderLid: !isJidGroup(key.remoteJid) ? senderLid : undefined,
            senderPn: !isJidGroup(key.remoteJid) ? senderPn : undefined,
            participant: isJidGroup(key.remoteJid) ? contextInfo?.participant : undefined,
            participantLid: isJidGroup(key.remoteJid) ? senderLid : undefined,
            participantPn: isJidGroup(key.remoteJid) ? senderPn : undefined,
        },
        chat: isJidGroup(key.remoteJid) ? key.remoteJid : contextInfo.participant,
        sender: contextInfo.participant,
        senderLid,
        senderPn,
        content,
        mtype: type,
        message: quotedMessage,
    };
};
//# sourceMappingURL=serialize.js.map